import 'package:sqflite/sqflite.dart';
import 'package:path/path.dart';

import '../models/book.dart';

class DatabaseHelper {
  static final DatabaseHelper _instance = DatabaseHelper._internal();
  factory DatabaseHelper() => _instance;
  DatabaseHelper._internal();

  static Database? _database;

  // Getter für die DB-Instanz (öffnet die DB bei erstmaligem Zugriff)
  Future<Database> get database async {
    if (_database != null) return _database!;
    _database = await _initDatabase();
    return _database!;
  }

  // Initialisiert die Datenbank
  Future<Database> _initDatabase() async {
    // Pfad zur Datenbank ermitteln
    final dbPath = await getDatabasesPath();
    final path = join(dbPath, 'books.db');
    // Öffnen (und erstellen falls nicht existent)
    return await openDatabase(
      path,
      version: 1,
      onCreate: _onCreate,
    );
  }

  // Wird beim erstmaligen Anlegen der DB aufgerufen, erstellt die Tabelle
  Future _onCreate(Database db, int version) async {
    await db.execute('''
      CREATE TABLE books(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        title TEXT,
        author TEXT,
        rating INTEGER,
        status TEXT,
        notes TEXT
      )
    ''');
  }

  // Neues Buch in DB einf\u00fcgen
  Future<int> insertBook(Book book) async {
    final db = await database;
    // Fuegt das Buch als Map ein und gibt die generierte ID zur\u00fcck
    int id = await db.insert(
      'books',
      book.toMap(),
      conflictAlgorithm: ConflictAlgorithm.replace,
    );
    return id;
  }

  // Alle Buecher aus der DB lesen
  Future<List<Book>> getBooks() async {
    final db = await database;
    // Alle Zeilen aus Tabelle auslesen
    final List<Map<String, dynamic>> maps = await db.query('books');
    // Map-Liste in Book-Objekte umwandeln
    return maps.map((map) => Book.fromMap(map)).toList();
    // Alternativ:
    // return List.generate(maps.length, (i) => Book.fromMap(maps[i]));
  }

  // Buch aktualisieren
  Future<int> updateBook(Book book) async {
    final db = await database;
    // Update anhand der ID
    return await db.update(
      'books',
      book.toMap(),
      where: 'id = ?',
      whereArgs: [book.id],
    );
  }

  // Buch loeschen
  Future<int> deleteBook(int id) async {
    final db = await database;
    return await db.delete(
      'books',
      where: 'id = ?',
      whereArgs: [id],
    );
  }
}
