import 'package:flutter/foundation.dart';
import '../data/database_helper.dart';
import '../models/book.dart';

class BookProvider extends ChangeNotifier {
  final DatabaseHelper _dbHelper = DatabaseHelper();
  List<Book> _books = [];

  List<Book> get books => _books; // Getter f\u00fcr die B\u00fccherliste

  // Bueher initial aus der DB laden
  Future<void> loadBooks() async {
    _books = await _dbHelper.getBooks();
    notifyListeners();  // UI benachrichtigen, dass Daten geladen wurden
  }

  // Neues Buch hinzufueggen
  Future<void> addBook(Book book) async {
    int id = await _dbHelper.insertBook(book);
    book.id = id;              // die vergebene ID setzen
    _books.add(book);
    notifyListeners();         // UI aktualisieren
  }

  // Existierendes Buch bearbeiten (aktualisieren)
  Future<void> updateBook(Book book) async {
    await _dbHelper.updateBook(book);
    // Lokale Liste aktualisieren: Book mit gleicher ID ersetzen
    int index = _books.indexWhere((b) => b.id == book.id);
    if (index != -1) {
      _books[index] = book;
    }
    notifyListeners();
  }

  // Buch loeschen
  Future<void> deleteBook(Book book) async {
    await _dbHelper.deleteBook(book.id!);
    _books.removeWhere((b) => b.id == book.id);
    notifyListeners();
  }
}
