import 'package:flutter/material.dart';
import 'package:provider/provider.dart';

import '../models/book.dart';
import '../providers/book_provider.dart';

class BookFormScreen extends StatefulWidget {
  final Book? book;
  const BookFormScreen({Key? key, this.book}) : super(key: key);

  @override
  _BookFormScreenState createState() => _BookFormScreenState();
}

class _BookFormScreenState extends State<BookFormScreen> {
  final _formKey = GlobalKey<FormState>();
  late TextEditingController _titleController;
  late TextEditingController _authorController;
  late TextEditingController _notesController;
  late int _rating;
  late String _status;

  @override
  void initState() {
    super.initState();
    // Initialwerte fuer Formularfelder setzen
    _titleController = TextEditingController(text: widget.book?.title ?? '');
    _authorController = TextEditingController(text: widget.book?.author ?? '');
    _notesController = TextEditingController(text: widget.book?.notes ?? '');
    _rating = widget.book?.rating ?? 3;            // Standard 3 Sterne
    _status = widget.book?.status ?? 'Geplant';    // Standard Status "Geplant"
  }

  @override
  void dispose() {
    // Controller freigeben
    _titleController.dispose();
    _authorController.dispose();
    _notesController.dispose();
    super.dispose();
  }

  void _saveBook() async {
    if (_formKey.currentState?.validate() ?? false) {
      // Formulardaten auslesen
      String title = _titleController.text.trim();
      String author = _authorController.text.trim();
      String notes = _notesController.text.trim();
      BookProvider bookProvider = Provider.of<BookProvider>(context, listen: false);

      if (widget.book == null) {
        // Neues Buch anlegen
        Book newBook = Book(title: title, author: author, rating: _rating, status: _status, notes: notes);
        await bookProvider.addBook(newBook);
      } else {
        // Bestehendes Buch updaten
        Book updatedBook = Book(
          id: widget.book!.id,
          title: title,
          author: author,
          rating: _rating,
          status: _status,
          notes: notes,
        );
        await bookProvider.updateBook(updatedBook);
      }
      Navigator.pop(context); // zurück zur Liste
    }
  }

  @override
  Widget build(BuildContext context) {
    final isEditing = widget.book != null;
    return Scaffold(
      appBar: AppBar(
        title: isEditing
            ? Hero(tag: 'title-${widget.book!.id}', child: Text(widget.book!.title))
            : const Text('Neues Buch hinzufügen'),
      ),
      body: Padding(
        padding: const EdgeInsets.all(16.0),
        child: Form(
          key: _formKey,
          child: SingleChildScrollView(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                // Titelfeld
                TextFormField(
                  controller: _titleController,
                  decoration: const InputDecoration(labelText: 'Titel'),
                  validator: (value) {
                    if (value == null || value.trim().isEmpty) {
                      return 'Bitte einen Buchtitel eingeben.';
                    }
                    return null;
                  },
                ),
                // Autorfeld
                TextFormField(
                  controller: _authorController,
                  decoration: const InputDecoration(labelText: 'Autor'),
                  validator: (value) {
                    if (value == null || value.trim().isEmpty) {
                      return 'Bitte einen Autor angeben.';
                    }
                    return null;
                  },
                ),
                // Bewertungsfeld (Dropdown 1-5)
                DropdownButtonFormField<int>(
                  value: _rating,
                  decoration: const InputDecoration(labelText: 'Bewertung'),
                  items: [1,2,3,4,5].map((val) {
                    return DropdownMenuItem(
                      value: val,
                      child: Text('$val Sterne'),
                    );
                  }).toList(),
                  onChanged: (val) {
                    if (val != null) setState(() { _rating = val; });
                  },
                ),
                // Statusfeld (Dropdown)
                DropdownButtonFormField<String>(
                  value: _status,
                  decoration: const InputDecoration(labelText: 'Status'),
                  items: <String>['Geplant', 'Lese ich gerade', 'Gelesen'].map((status) {
                    return DropdownMenuItem(
                      value: status,
                      child: Text(status),
                    );
                  }).toList(),
                  onChanged: (val) {
                    if (val != null) setState(() { _status = val; });
                  },
                ),
                // Notizfeld
                TextFormField(
                  controller: _notesController,
                  decoration: const InputDecoration(labelText: 'Notizen'),
                  maxLines: 4,
                ),
                const SizedBox(height: 20),
                Center(
                  child: ElevatedButton(
                    onPressed: _saveBook,
                    child: Text(isEditing ? 'Änderungen speichern' : 'Buch hinzufügen'),
                  ),
                ),
              ],
            ),
          ),
        ),
      ),
    );
  }
}
