import 'package:flutter/material.dart';
import 'package:provider/provider.dart';

import '../models/book.dart';
import '../providers/book_provider.dart';
import '../screens/book_form_screen.dart';

class BookListScreen extends StatefulWidget {
  const BookListScreen({Key? key}) : super(key: key);
  @override
  _BookListScreenState createState() => _BookListScreenState();
}

class _BookListScreenState extends State<BookListScreen> {
  String _filterStatus = 'Alle'; // aktueller Filter (Alle/Geplant/Lese ich gerade/Gelesen)

  @override
  void initState() {
    super.initState();
    // Bücher aus der DB laden, wenn der Screen aufgebaut wird
    Provider.of<BookProvider>(context, listen: false).loadBooks();
  }

  @override
  Widget build(BuildContext context) {
    // Über den Provider die Bücherliste beziehen
    final bookProvider = Provider.of<BookProvider>(context);
    final allBooks = bookProvider.books;
    // Filter anwenden:
    List<Book> displayedBooks;
    if (_filterStatus == 'Alle') {
      displayedBooks = allBooks;
    } else {
      displayedBooks = allBooks.where((book) => book.status == _filterStatus).toList();
    }

    return Scaffold(
      appBar: AppBar(
        title: const Text('Mein Bücherregal'),
      ),
      body: Column(
        children: [
          // Filter Dropdown
          Padding(
            padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 8),
            child: Row(
              children: [
                const Text('Filter: ', style: TextStyle(fontWeight: FontWeight.bold)),
                DropdownButton<String>(
                  value: _filterStatus,
                  items: <String>['Alle', 'Geplant', 'Lese ich gerade', 'Gelesen']
                      .map((status) => DropdownMenuItem(
                    value: status,
                    child: Text(status),
                  ))
                      .toList(),
                  onChanged: (value) {
                    if (value != null) {
                      setState(() {
                        _filterStatus = value;
                      });
                    }
                  },
                ),
              ],
            ),
          ),
          // Liste der Bücher
          Expanded(
            child: displayedBooks.isEmpty
                ? Center(
              child: Text(
                'Keine Bücher vorhanden.',
                style: Theme.of(context).textTheme.titleSmall,
              ),
            )
                : ListView.builder(
              itemCount: displayedBooks.length,
              itemBuilder: (context, index) {
                Book book = displayedBooks[index];
                return Card(
                  margin: const EdgeInsets.symmetric(horizontal: 12, vertical: 6),
                  child: ListTile(
                    title: Hero(
                      tag: 'title-${book.id}',
                      child: Text(book.title,
                          style: TextStyle(fontSize: 16, fontWeight: FontWeight.bold)),
                    ),
                    subtitle: Text('${book.author} – ${book.status}'),
                    trailing: Row(
                      mainAxisSize: MainAxisSize.min,
                      children: List.generate(5, (i) {
                        return Icon(
                          i < book.rating ? Icons.star : Icons.star_border,
                          color: Colors.amber,
                          size: 20,
                        );
                      }),
                    ),
                    onTap: () {
                      // Detailansicht \u00f6ffnen (Buch bearbeiten)
                      Navigator.push(
                        context,
                        MaterialPageRoute(
                          builder: (_) => BookFormScreen(book: book),
                        ),
                      );
                    },
                  ),
                );
              },
            ),
          ),
        ],
      ),
      floatingActionButton: FloatingActionButton(
        onPressed: () {
          // Navigiere zur Formularansicht zum Hinzuf\u00fcgen eines neuen Buchs
          Navigator.push(
            context,
            MaterialPageRoute(builder: (_) => const BookFormScreen()),
          );
        },
        child: const Icon(Icons.add),
        tooltip: 'Neues Buch hinzufügen',
      ),
    );
  }
}
